; Copyright (c) 1998-2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_LOX_SPECTRA
;
;LAST CHANGED: ----------------------------------- 16-oct-05 (aph)
;
; PURPOSE:
;	This function reada a Lox spectral file
;
; CATEGORY: spectral input
; called from READ~spectra~Lox
;
; CALLING SEQUENCE:
;	Result = READ_LOX_SPECTRA(file=file, filter=filter, NOF=nof, $
;             print_source = print_source, _extra=e)
;
; INPUTS:
;	All input parameters are passed as keywords.
;
; KEYWORDS:
;	FILE:	filename
;	FILTER:	optional user selectable filter (default = '*.'
;	NOF:	no filter (*.*)
;	PRINT_SOURCE:	if set to 1; print input file as it is read
;	_EXTRA_E:	forward other parameters
;
; OUTPUTS: none
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (17-may-04 aph) adapt from pem_load fle
; (01-jun-04 aph) adapt for stack files without </spectra>
; (30-jan-05 aph) add sorting to make compatible with reverse scanning
; (14-jun-05 aph) correct spectral read in to get correct number of points
; (16-jun-05 aph) add kluge to handle blank line in jun-05 lox_spectra format
; (16-oct-05 aph) CHANGED NAME (from Read_lox  to Read_lox_spectra)
;-

Function read_lox_spectra, file=file, filter=filter, NOF=nof, print_source = print_source, _extra=e
 on_error,2
 @axis_com

if not keyword_set(file) then begin
	 fltr='*.lox'
	 if keyword_set(nof) then fltr='*.*'
	 if keyword_set(filter) then fltr=filter
	 file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath, $
	       title = 'Select Lox energy file')
endif
s = 0
if strlen(file) LE 0 THEN RETURN,s  ; bail-out if no filename

openr, iunit, file, /get_lun

on_ioerror, end_of_file
test = ''
while test NE '<spectra>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if keyword_set(print_source) then print, test
endwhile

np = 0
x = 0.  &  y = 0.
while test NE '</spectra>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if test EQ '' then begin		; adapt to change in format 14-jun-05
		readf, iunit, test
		test = strtrim(test, 2)
	endif
	if keyword_set(print_source) then print, test
	if test NE '</spectra>' then begin
		x = [x,0.] & y = [y,0.]
		np = np + 1
		reads, test, xt, yt
		x(np) = xt  & y(np) = yt
	endif
endwhile
; -------- remove last point - an extra (0,0)
;np=np-1
x = x(1:np)
y = y(1:np)


; ------- sort data and correct any non-monotonic errors (2 identical E's)
d = {x:x, d:y}
d = ax_sort_mono(d, /axis)
x = d.x  & y = d.d

name_bits = ax_name(file)
fileshort = name_bits(1)
s={t:'1d', d: y(0:np-1), x: x(0:np-1), xl: 'Energy (eV)', dn:y(0:np-1), dl: fileshort}
print, np,' point Lox spectrum read from ', file

close, iunit & free_lun, iunit

return, s

end_of_file:
; -------------------------------- check for stack; if so read from image energies --------------
close, iunit & free_lun, iunit
openr, iunit, file, /get_lun
on_ioerror, end_of_file2

print, 'Not a standard Lox spectral file; try to read a stack'

while test NE '<stack>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if keyword_set(print_source) then print, test
endwhile

np = 0
x = 0.  &  y = 0.

while test NE '</stack>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if keyword_set(print_source) then print, test
	t = strpos(test,'energy="')
	test2 = strmid(test, t+8)
	t = strpos(test2,'"')
	Estring=strmid(test2, 0, t)
	x(np) = float(Estring)
	if keyword_set(print_source) then print, Estring, x(np)
	y(np) = x(np)
	x = [x,0.] & y = [y,0.]
	np = np + 1
endwhile
np = np-1

name_bits = ax_name(file)
fileshort = name_bits(1)
s={t:'1d', d: y(0:np-1), x: x(0:np-1), xl: 'Energy (eV)', dn:y(0:np-1), dl: fileshort}
print, np,' point Lox spectrum read from ', file

close, iunit & free_lun, iunit

return, s

end_of_file2:
	print, 'Not a Lox spectral file'
	return, s
end