; Copyright (c) 1998-2005 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_LOX_SPECTRA
;
;LAST CHANGED: ----------------------------------- 13-nov-05 (aph)
;
; PURPOSE:
;	This function reada a Lox spectral file into an aXis 1-d structure
; or into a multi-column array
;
; CATEGORY: spectral input
; called from READ~spectra~Lox
;
; CALLING SEQUENCE:
;	Result = READ_LOX_SPECTRA(file=file, filter=filter, NOF=nof,  multi = multi, $
;            channel=channel, region = region, verbose = verbose, _extra=e)
;
; INPUTS:
;	All input parameters are passed as keywords.
;
; KEYWORDS:
;	FILE:			filename
;	FILTER:			optional user selectable filter (default = '*.lox'
;	NOF:			no filter (*.*)
;	MULTI 			read all spectra in multi-column file
;	VERBOSE:		if set to 1; print input file as it is read
;	CHANNEL			read in specific channel of a multi-column file
;	REGION			read-in from file associated with a specific region (now = PEEM-channel)
;	SILENT			if set, no feedback in print or axis_log
;	_EXTRA_E:		forward other parameters to any programs this one calls
;
; OUTPUTS: returns a structure containing spectrum, unless MULTI keyword is set
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;
; MODIFICATION HISTORY:
; (17-may-04 aph) adapt from pem_load fle
; (01-jun-04 aph) adapt for stack files without </spectra>
; (30-jan-05 aph) add sorting to make compatible with reverse scanning
; (14-jun-05 aph) correct spectral read in to get correct number of points
; (16-jun-05 aph) add kluge to handle blank line in jun-05 lox_spectra format
; (16-oct-05 aph) CHANGED NAME (from Read_lox  to Read_lox_spectra)
;             set up to do multi-column reading for regions
; (13-Nov-05 aph) read in mesh or specific region signal
;-

Function read_lox_spectra, file=file, filter=filter, NOF=nof, multi = multi, $
       channel=channel, region = region, verbose = verbose, silent = silent,  _extra=e
 on_error,2
 @axis_com

if not keyword_set(file) then begin
	 fltr='*.lox'
	 if keyword_set(nof) then fltr='*.*'
	 if keyword_set(filter) then fltr=filter
	 file = pickfile2(/READ, FILTER=fltr, /LPATH, DEFPATH=defpath, $
	       title = 'Select Lox energy file')
endif
s = 0
if strlen(file) LE 0 THEN RETURN,s  ; bail-out if no filename

openr, iunit, file, /get_lun

; on_ioerror, end_of_file
test = ''
while test NE '<spectra>' do begin
	readf, iunit, test
	test = strtrim(test,2)
	if keyword_set(verbose) then print, test
endwhile
; -- now at start of multi-column spectral data
; -- determine how many columns there are from the first line
readf, iunit, test
t=parsestring(test) & nt = size(t)
n_entries = nt(1)
nlines=6000 			; hopefully larger than any Lox scan !
ans=fltarr(n_entries, nlines)
ans(*,0)= t
j=long(1)
while test NE '</spectra>' and not eof(iunit) do begin
   ;if j/100 eq j/100. then print,j
   readf,iunit, test
   test = strtrim(test,2)
   if keyword_set(verbose) then print, test
   if test NE '</spectra>' then begin
   		t=parsestring(test)
		ans(*,j)=t
   		j=j+1
   	endif
endwhile

ans = ans(*,0:j-1)		; KLUGE!! - the trip on </spectra> at end is not working !

close, iunit & free_lun, iunit
if keyword_set(multi) then return, ans

nt = size(ans) & np =nt(2)
name_bits = ax_name(file)
fileshort = name_bits(1)
text = Strtrim(string(fix(np)),2) + ' point Lox spectrum read from ' + fileshort
if NOT keyword_set(silent) then axis_log, text



; ------- sort data and correct any non-monotonic errors (2 identical E's)
x = ans(0,*)
y = ans(1,*)
; for i = 0, np-1 do print, x(i),y(i)
d = {x:x, d:y}
d = ax_sort_mono(d, /axis)
x = d.x  & y = d.d
s={t:'1d', d: y(0:np-1), x: x(0:np-1), xl: 'Energy (eV)', dn:y(0:np-1), dl: fileshort}


return, s

end_of_file:
; -------------------------------- check for stack; if so read from image energies --------------
close, iunit & free_lun, iunit

openr, iunit, file, /get_lun
on_ioerror, end_of_file2

axis_log, 'Not a standard Lox spectral file; try to read as a stack'

while test NE '<stack>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if keyword_set(verbose) then print, test
endwhile

np = 0
x = 0.  &  y = 0.

while test NE '</stack>' do begin
	readf, iunit, test
	test = strtrim(test, 2)
	if keyword_set(verbose) then print, test
	t = strpos(test,'energy="')
	test2 = strmid(test, t+8)
	t = strpos(test2,'"')
	Estring=strmid(test2, 0, t)
	x(np) = float(Estring)
	if keyword_set(verbose) then print, Estring, x(np)
	y(np) = x(np)
	x = [x,0.] & y = [y,0.]
	np = np + 1
endwhile
np = np-1

name_bits = ax_name(file)
fileshort = name_bits(1)
s={t:'1d', d: y(0:np-1), x: x(0:np-1), xl: 'Energy (eV)', dn:y(0:np-1), dl: fileshort}
if NOT keyword_set(silent) then axis_log, np,' point Lox spectrum read from ', file

close, iunit & free_lun, iunit

return, s

end_of_file2:
	axis_log, 'Not a Lox spectral file'
	return, s
end