; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	  STACK_FIT
;
;LAST CHANGED: ----------------------------------- 12-aug-01
;
;PURPOSE:
;	This procedure selects inputs for fitting the spectrum at each pixel
; in a stack to sum of model spectra
; It uses COMMON blocks to store and transfer data
;
;CATEGORY:
;	AXIS:  stack analysis (operates stand-alone as well)
;
;CALLING SEQUENCE:
; for procedures:
;	STACK_FIT(/AXIS,/verbose)
;
;CALLED FROM AXIS:
;	->Stacks->stack_fit
;
;INPUTS:
;	All input parameters are obtained through user dialogs.
;
;KEYWORDS:
;	AXIS - if set, indicates called from AXIS
;	VERBOSE - if set, displays results as generated line-by-line (SLOW!)
;
;ROUTINES
;	STACK_FIT - main routine to execute stack fit
;	GROOM1 - linear fit to pre-edge values
;	INPUTS to GROOM1
;	spectrum - the input and the result after subtraction of back
;	back     - the constant to be subtracted
;	integral - integrated value in background region
;	zero     - number of points to evaluate constant background
;
;OUTPUTS:
;	If run from AXIS, the results are placed in AXIS buffers
; buffer 1 - n are the spectra of the n-models to be used in the fit,
;              interpolated to the energy values of the stack.
; buffer 4 - (4+n) are the resulting composition maps for the n components
; buffer 8 - linear term of fit
; buffer 9 - chi-square of the fit
; otherwise display windows are generated as required
;
;COMMON BLOCKS:
;	@AXIS_COM	standard set of common blocks
;	@ANALCOM  - STACK (cjj)
;	Volume_data - STACK
; 	@BSIF_COM - nc data
;
;PROCEDURE:
;	After reading in the stack (*.nc) and the model spectra (*.txt)
; and the names to be used for the output files,
; the spectrum at each pixel is fit to (linear) + SUM{coeff(i)*model spectra(i)}
; The coefficients at each pixel constitute the component map for species i.
; The user can choose to perform a pre-edge subtraction based on the first {m} -data points
; in order to base the analysis on solely the core spectra signal
;
;MODIFICATION HISTORY:
; ***********
; ORIGINAL kernel of this routine was written by Rick Kneedler
; ***********
; (25-May-99 aph) first developed from Kneedler code
; (19-Jun-99 aph) allow user choice to background subtract image spectra
; (07-jul-99 aph) extend to arbitrary number of components
; (09-sep-99 aph) store all results automatically; correct (x,y) scale stuff
; (20-sep-99 aph) remove linear term; set NaN, Inf to zero; integer col #
; (23-oct-99 aph) force plot of components on store
; ( 7-oct-99 aph) allow assignment of simple names to output files
; (26-nov-99 aph) get rid of error message for text prompt
; (23-Dec-99 aph) correct bad point filter on output component maps
; (27-feb-00 aph) add groupIP to get_text cal; axis standard documentation
; (09-apr-00 aph) change groom.pro to fit a sloped line with feedback
; (19-oct-00 aph) increased max # of components to 8
; (13-nov-00 aph) fixed error for GE 5 component; model data
; (09-jan-01 aph) move groom to run with compile-on-fly
; (12-aug-01 aph) put overwrite checking into file writing procedure
;-

;****************************************************
pro groom, energy, spectrum, nzero, line, back, integral
;
; ------------------ last changed: 07-apr-00 (aph)
;
; procedure by rk 4/99 to groom one spectrum.
; Outputs bkd-subtracted spectrum, background, and integral of line

;---------- fit first nzero+1 points are fit to a linear equation to get the background
line = POLY_FIT(energy(0:nzero), spectrum(0:nzero), 1 )
back = line(0) + energy*line(1)
spectrum = spectrum - back
; ----- report back info re relative amount of background
back = total(back)
integral=total(spectrum)
return
end
; ****************************************************

PRO stack_fit, axis=axis, verbose = verbose

@axis_com
@analcom
COMMON volume_data, image_stack
@bsif_com

on_error,2

; read-in stack data
file = pickfile2(/READ, FILTER='*.ncb', /LPATH)
if strlen(file) GT 0 THEN BEGIN  ; bail-out if no filename
	stack_rb, file		; stack in image_stack (Volume_data), energies in ev (ANALCOM)
	t=ax_name(file)
	fileroot = t(0) + t(1)
	fileroot = get_text(prompt = 'Root name for output files',val=fileroot, group = axis_ID)
	fileroot = fileroot + '_'

; read in reference spectra data
	if  keyword_set(AXIS) THEN BEGIN
		n_comp = get_num(Prompt= '# of components (1-8)', val=3, group = axis_id)
	endif else n_comp = get_num(Prompt= '# of components (1-8)', val=3)
	model = fltarr(n_comp, n_elements(ev))
	comp_lbl = strarr(n_comp)
	for i = 0, n_comp-1 do begin
		comp_file = pickfile2(/read, Filter='*.txt')
		if comp_file EQ '' then return  ; cancel at any time
		tmp = spc_load(filter='*.txt',file=comp_file)

; --- establish name of component for file storage and header
		t = ax_name(comp_file)
		comp_lbl(i) = t(1)
		text = 'Name for component '+ strtrim(string(i),2)
		if keyword_set(axis) then begin
			comp_lbl(i) = get_text(prompt = text, val = comp_lbl(i),group = axis_id)
		endif else comp_lbl(i) = get_text(prompt = text, val = comp_lbl(i), group = axisID)

; interpolate to the same E-scale as the stack
; check that range of reference data contains range of ev
; ----------- to write ---------------------

; interpolate and store in AXIS data buffer
		model(i,*)=interpol(tmp.d,tmp.x,ev)
		tmp = {t:'1d', d: model(i,*), x: ev, xl: 'X', dn:model(i,*), dl: 'interp '+ tmp.dl}
		CurBuf = i+1
		if  n_tags(tmp) NE 0 AND keyword_set(AXIS) THEN BEGIN
			HANDLE_VALUE, Data(Curbuf), tmp, /SET
			PlotBuf, Curbuf
		endif
	endfor

; -------START OF KNEEDLER's FITSTACK procedure ---------

; determine number of components
;	mod_size = size(model)
;	n_comp = fix(mod_size(1))
	print, 'Fitting spectrum at each pixel to ', strtrim(string(n_comp),2),' components'
	; and # of energies (images)
	n_E = n_elements(ev)

; groom reference spectra to have zero background (but they are
; assumed be normalized to each other in some way)
	if  keyword_set(AXIS) THEN BEGIN
		nzero = get_num(Prompt='#bkground points for model spectra',Val = 0, group = axis_id)
		nzero_img = get_num(Prompt='#bkground points for image spectra?', Val=0, group = axis_id)
	endif else begin
		nzero = get_num(Prompt='#bkground points for model spectra',Val = 0)
		nzero_img = get_num(Prompt='#bkground points for image spectra?', Val=0)
	endelse
	Widget_control, /hourglass
	if nzero GT 0 then begin
		for i = 0, n_comp-1 do begin
			tmp = reform(model(i,*))
			groom, ev, tmp, nzero, line, bak, inte
			model(i,*) = tmp
			print,'reference spectrum ', strcompress(string(i+1)), $
			   '; integrated intensity = ',inte, ' : subtracted intensity =', bak
			print, '    Linear fit to background: slope = ', line(1), ' intercept = ', line(0)
		endfor
	ENDIF

; switch plotting depending on keyword AXIS
	if NOT keyword_set(axis) then begin
	; plot reference signals
		nwin,0
		wset,0
		plot, ev,model(0,*)
		for i = 1, n_comp-1 do oplot,ev,model(i,*)
;		oplot, ev, 10*(line/max(line))
	endif

; ------------ set up (x,y) scales from bsif_com information
	x_step = (x_stop - x_start)/n_cols
	y_step = (y_stop - y_start)/n_rows
	img_x = x_start + findgen(n_cols)*x_step
	img_y = y_start + findgen(n_rows)*x_step

; --------- set-up output arrays
	const=fltarr(n_cols,n_rows)
	comp = fltarr(n_comp, n_cols,n_rows)
	chisq=const
	bk=const
	int=const
	linear=const
	if nzero_img GT 0 then begin  ; save the line fits to background
		slope = bk
		intercept = bk
	endif
	residual = float(image_stack)
	if NOT keyword_set(axis) then begin
		window,1,xpos=0,ypos=200,xsize=n_cols,ysize=n_rows
		window,2,xpos=0,ypos=400,xsize=n_cols,ysize=n_rows
		if n_comp EQ 3 then window,3,xpos=0,ypos=600,xsize=n_cols,ysize=n_rows
		window,4,xpos=0,ypos=0,xsize=n_cols,ysize=n_rows
		loadct,0
	endif

; set up regression fit params: x is array of model spectra ; -- removed(& line)
;  unweighted (weightsi=1, use /relative_weights keyword)
	if n_comp GE 1 then tmp0 = transpose(reform(model(0,*)))
	if n_comp GE 2 then tmp1 = transpose(reform(model(1,*)))
	if n_comp GE 3 then tmp2 = transpose(reform(model(2,*)))
	if n_comp GE 4 then tmp3 = transpose(reform(model(3,*)))
	if n_comp GE 5 then tmp4 = transpose(reform(model(4,*)))
	if n_comp GE 6 then tmp5 = transpose(reform(model(5,*)))
	if n_comp GE 7 then tmp6 = transpose(reform(model(6,*)))
	if n_comp GE 8 then tmp7 = transpose(reform(model(7,*)))
	if n_comp EQ 1 then x=[tmp0]  ;,transpose(line)]
	if n_comp EQ 2 then x=[tmp0,tmp1]   ; ,transpose(line)]
	if n_comp EQ 3 then x=[tmp0,tmp1,tmp2] ;,transpose(line)]
	if n_comp EQ 4 then x=[tmp0,tmp1,tmp2,tmp3]  ;,transpose(line)]
	if n_comp EQ 5 then x=[tmp0,tmp1,tmp2,tmp3,tmp4]  ;,transpose(line)]
	if n_comp EQ 6 then x=[tmp0,tmp1,tmp2,tmp3,tmp4,tmp5]  ;,transpose(line)]
	if n_comp EQ 7 then x=[tmp0,tmp1,tmp2,tmp3,tmp4,tmp5,tmp6]  ;,transpose(line)]
	if n_comp EQ 8 then x=[tmp0,tmp1,tmp2,tmp3,tmp4,tmp5,tmp6,tmp7]  ;,transpose(line)]
	wts = make_array(n_E,value=1.0)
	bak = 1.  & inte = 1.
	for i=0,n_cols-1 do begin
	  	for j=0,n_rows-1 do begin
			spectrum=reform(image_stack(i,j,*))
			if nzero_img GT 0 then begin
				groom, ev, spectrum, nzero_img, line, bak, inte
				intercept(i,j) = line(0)
				slope(i,j) = line(1)
			endif
			bk(i,j)=bak
			int(i,j)=inte
			fit=regress(x, spectrum, wts, yfit, c, z, z, z, z, chi2, status, /relative_weight)
			for k = 0, n_comp -1 do begin
				comp(k,i,j)=fit(k)
			endfor
;			linear(i,j)=fit(n_comp)
			const(i,j)=c
			chisq(i,j)=chi2
			residual(i,j,*) = image_stack(i,j,*) - yfit
		;	wset,0
		;	plot,ev,spectrum
		;	oplot,ev,yfit
		;   print,fit,const(i,j),chisq(i,j),bk(i,j),int(i,j)
	  	endfor

; ------- user feedback ... working ....
		if keyword_set(AXIS) THEN BEGIN
			WIDGET_CONTROL, Uprompt, SET_VALUE='STACK_FIT: line ' $
			    + strcompress(string(i)) +' of ' + strcompress(string(fix(n_cols)))
		endif


		if NOT keyword_set(axis) and keyword_set(verbose) then begin
;	NB the following ONLY works on graphics display devices with 3-channel (RBG) implemented
; plot results line-by-line   component 1 is red, 2 is green, 3 is blue
		  	top=max(comp)
			comp(*,0,0)=top

			if n_comp GE 1 then begin
				wset,1
				tvscl,reform(comp(0,*,*)),0,0,1
			endif
			if n_comp GE 2 then begin
				wset,2
				tvscl,reform(comp(1,*,*)),0,0,2
			endif
	  		if n_comp GE 3 then begin
	  			wset,3
	  			tvscl,reform(comp(2,*,*)),0,0,3
	  		endif
	  		if n_comp GE 4 then begin
	  			wset,4
	  			tvscl,reform(comp(3,*,*)),0,0,3
	  		endif
  			if n_comp GE 5 then begin
	  			wset,5
	  			tvscl,reform(comp(4,*,*)),0,0,3
	  		endif
	  		if n_comp GE 6 then begin
	  			wset,5
	  			tvscl,reform(comp(5,*,*)),0,0,3
	  		endif
	  		if n_comp GE 7 then begin
	  			wset,5
	  			tvscl,reform(comp(6,*,*)),0,0,3
	  		endif
	  		if n_comp GE 8 then begin
	  			wset,5
	  			tvscl,reform(comp(7,*,*)),0,0,3
	  		endif
; Attempt to generate 3-color composite
	  		wset,n_comp+1
	  		device, get_graphics_function= oldg,SET_GRAPHICS_FUNCTION = 7 ; set to "OR"
			if n_comp GE 1 then tvscl,reform(comp(0,*,*)),0,0,1
			if n_comp GE 2 then tvscl,reform(comp(1,*,*)),0,0,2
	  		if n_comp ge 3 then tvscl,reform(comp(2,*,*)),0,0,3
	  		device, set_graphics_function= oldg
		endif
	; component 1 is green, component 2 is violet
;		comp1 = reform(comp(0,*,*))
;		comp2 = reform(comp(1,*,*))
;		c_max = max([comp1,comp2],min=c_min)
	;	print, c_max, c_min
;		wset,1
;		tvscl,comp1        		; *256/c_max+c_min + 1  (failed attempt to give constant scale)
;		wset,2
;		tvscl,comp2,0,0,2        ; *256/c_max+c_min + 1

	endfor

; --------- provide information about baclground fitting if done on data
	if nzero_img GT 0 then begin
		print, 'Background over image pixels'
		print, 'Average slope = ', total(slope)/n_elements(slope), ' +/-', variance(slope)
		print, 'Average intercept = ', total(intercept)/n_elements(intercept),' +/-', variance(intercept)
	endif

; finished the fit - store and display residuals
	if keyword_set(axis) then begin
;		tmp = {t:'2d', d: linear, x: img_x ,y: img_y, xl: 'X', yl: 'Y',  dl: 'linear '}
;		tmp = ax_order(tmp)
;		HANDLE_VALUE, Data(8), tmp, /SET
;		file = fileroot+'linear.axb'
;		test = axb_save(tmp,file=file)
; -------- constant term in buffer 8----
; ----------- replace NaN values with 0.
		tmp = {t:'2d', d: const, x: img_x ,y: img_y, xl: 'X', yl: 'Y',  dl: 'constant '}
		bad_index = where(finite(tmp.d) EQ 0, count)
		if count GT 0 then begin
			tmp.d(bad_index) = 0.
			print, 'stack_fit: WARNING - NaN or Inf pixels set to 0.'
		endif
		HANDLE_VALUE, Data(8), tmp, /SET
		file = fileroot+'constant.axb'
		overwrite_all = 0   ; reset overwrite flag in case was set by earlier uses
		test = axb_save(tmp,file=file)
		PlotBuf,8
; -----------  store results in buffers and thumbnails
		for i = 0, n_comp-1 do begin
			tmp = {t:'2d', d: reform(comp(i,*,*)), x: img_x ,y: img_y, xl: 'X', $
			         yl: 'Y',  dl: comp_lbl(i)}
			bad_index = where(finite(tmp.d) EQ 0, count)
			if count GT 0 then begin
				tmp.d(bad_index) = 0.
				print, 'stack_fit: WARNING - NaN or Inf pixels set to 0.'
			endif
			tmp = ax_order(tmp)
			if n_comp EQ 4 then offset = 5 else offset = 4
			if n_comp GT 4 then offset = 1
			CurBuf = i + offset
			HANDLE_VALUE, Data(Curbuf), tmp, /SET
			Plotbuf, CurBuf
			file = fileroot + comp_lbl(i) + '.axb'
			test = findfile(file)
			if test(0) EQ file AND overwrite_all NE 1 then begin
			    t = file_overwrite(group = axis_ID, val = 1)
			    if t EQ 0 then file = pickfile2(/write, LFILE = file)
			    if t EQ 2 then overwrite_all = 1
			endif
			test = axb_save(tmp,file=file)
		endfor
		tmp = {t:'2d', d: chisq, x: img_x ,y: img_y, xl: 'X', yl: 'Y',  dl: 'chi-square '}
		bad_index = where(finite(tmp.d) EQ 0, count)
		if count GT 0 then begin
			tmp.d(bad_index) = 0.
			print, 'stack_fit: WARNING - NaN or Inf pixels set to 0.'
		endif
		tmp = ax_order(tmp)
		HANDLE_VALUE, Data(9), tmp, /SET
		file = fileroot + 'chi.axb'
		test = findfile(file)
		if test(0) EQ file AND overwrite_all NE 1 then begin
		    t = file_overwrite(group = axis_ID, val = 1)
		    if t EQ 0 then file = pickfile2(/write, LFILE = file)
		    if t EQ 2 then overwrite_all = 1
		endif
		test = axb_save(tmp,file=file)
		CurBuf = 9
		PlotBuf,CurBuf
; ----------- store residual ?
		WIDGET_CONTROL, Uprompt, SET_VALUE='save residuals stack'
		file = pickfile2(/WRITE, FILTER='*.ncb', /LPATH)
		if strlen(file) GT 0 then begin
			image_stack = residual
			stack_wb, file		; store for later viewing
		endif
; --------- store fitted background ?
		if nzero_img GT 0 then begin
			WIDGET_CONTROL, Uprompt, SET_VALUE='save background stack'
			file = pickfile2(/WRITE, FILTER='*.ncb', /LPATH)
			if strlen(file) GT 0 then begin
				for i = 0, n_E-1 do image_stack(*,*,i) = intercept + slope*eV(i)
				stack_wb, file
			endif
		endif
	endif
; ------------- display residuals
	if NOT keyword_set(axis) then begin
		d_max=float(max(image_stack))
		window,5,xpos=200,ypos=200,xsize=n_cols,ysize=n_rows
		window,6,xpos=200,ypos=400,xsize=n_cols,ysize=n_rows
		for i=0,n_E-1 do begin
		;	print, ev(i)
			wset,n_comp + 2
		;	tv,500*residual,*,*,i)/image_stack(*,*,i)*256./d_max + 1
			tvscl,residual(*,*,i)     ; /image_stack(*,*,i)
			wset,n_comp + 3
		;	tv,image_stack(*,*,i)*256./d_max + 1
			tvscl,image_stack(*,*,i)
		;	wait,0.1
		endfor
	endif
endif
END


