 ; Copyright (c) 1998-2001 A.P. Hitchcock  All rights reserved
;+
;NAME: READ_SDFE
;
;LAST CHANGED: ----------------------------------- 22-aug-01
;
; PURPOSE:
;	This function reads self-defining format ascii files used in ALS BL 5.3.2 STXM.
; Images, stacks and linescans have *.hdr and a set of *.xim files
; for {energies.channels.regions}
; Spectra are a single *.xsp file containing header and multi-column spectra.
;
; CATEGORY:
;	Input / output utilty
;
; CALLING SEQUENCE:
;	Result = READ_SDF(file, nof=nof, filter=filter, verbose=verbose, groupID=groupID, header_only=header_only, _extra=e)
;
; INPUTS:
; 	FILE	optional file name
;
; KEYWORDS:
;	FILTER	   user defined filter
;	VERBOSE		print out details of parsing structure (testing)
;	CHANNEL		preselected data channel
;	GROUPID		group leader (Axis_ID if called from aXis2000 and axis)
;	HEADER_ONLY print header contents only
;	_EXTRA		other passed through parameters
;
; OUTPUTS:
;	structure variable; either 1d (spectra) or 2d (image)
;
; COMMON BLOCKS:
;	AXIS_COM	standard set of common blocks
;	SDFORMAT	reference number for naming muti-element arrays of structures
;
; MODIFICATION HISTORY:
; (11-feb-01 aph) first version
; (23-feb-01 aph) arrays of structures implemented
; (25-feb-01 aph) option to return structure only implemented
; (22-aug-01 aph) rescued from axis2000-1b (Jul-01) for early file readin
;-

function read_sdfe, file, filter=filter, channel=channel, verbose=verbose, groupID=groupID, header_only=header_only, _extra=e
on_error,2
@axis_com
common sdformat, starnum

t = size(starnum)
; print, t
if t(1) EQ 0 then starnum = 0   ; start of unique names - SHOULD USE A CHECK for existence
; print, 'starnum ',  starnum

if n_elements(file) eq 0 then begin  ;popup file dialog box
   if not keyword_set(filter) then fltr='*.hdr' else fltr = filter
   file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif
s = 0
if strlen(file) LE 0 THEN return, s  ; bail-out if no filename
; ------- Open the header to be read -------
; first force to be a header
t = ax_name(file)
fileshort = t(1)
file_hdr = t(0) + t(1) + '.hdr'
t = findfile(file_hdr)  & t = t(0)
IF t EQ '' then begin
	s = read_bl5(file, group = Axis_ID)		; kluge until spectra format split
	return, s
ENDIF
; -----------
on_ioerror,nofile
get_lun,lun
openr,lun,file_hdr
; ------- Read header file into a single string (max = 32600 characters)  --------
line = ' '
all = ''
count = 0
while not  eof(lun) do begin
	readf,lun, line
;		print, line
	all = all + line
	count = count + 1			; the number of lines is used in spectral readin
endwhile
close, lun

; format for recursive structure parsing
all = '{'+all+'};'
; parse the structure
if keyword_set(verbose) then ScanDef = parse_sdf(all,/verbose) $
   else ScanDef = parse_sdf(all)
; help, /struct, ScanDef.scandefinition

if keyword_set(header_only) then begin
; ------- print header information
;	XDisplayFile,file_hdr
	return, ScanDef
endif

; --------- Read in data according to the type of scan -----
n_channels = n_elements(ScanDef.ScanDefinition.channels)
n_data =-1
if channel EQ 0 then begin
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = '5.3.2 data from '+ fileshort
	for i = 0, n_channels-1 do begin
		text = string(i+1) + '  ' + ScanDef.ScanDefinition.channels[i].name
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	endfor
	while n_data LE 0 OR n_data GT n_channels do begin
		if keyword_set(groupID) then $
		  n_data = get_num(prompt='Choose data channel', val=2, group=groupID) $
		   else n_data = get_num(prompt='Choose data channel', val=2)
	endwhile
endif else 	n_data = channel
Print, file, '   # data channels', n_channels
case n_data of
	1 : let = 'a'
	2 : let = 'b'
	3 : let = 'c'
	4 : let = 'd'
	5 : let = 'e'
else : print, 'Only set up for 5 channels max'
endcase
TYPE = ScanDef.ScanDefinition.type
CASE TYPE of
	'Image Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		xl = string(FORMAT='("x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'um  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," image read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'OSA Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.OsaScan.DwellTime
		xl = string(FORMAT='("OSA x (um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'OSA y (um)  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," OSA scan read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(group) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'Focus Scan' : BEGIN
		x = float(ScanDef.ScanDefinition.PAxis.points)
		nx = n_elements(x)
		y = float(ScanDef.ScanDefinition.QAxis.points)
		ny = n_elements(y)
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = text_read1(img_file)
;	------ pack into AXIS 2d structure -----------------
		energy = ScanDef.ScanDefinition.StackAxis.min
		dwell = ScanDef.FocusScan.DwellTime
		xl = string(FORMAT='("(um)     E = ",f8.3," eV     dwell = ",f5.2," ms")', $
     			 energy, dwell)
		t = ax_name(file)
		fileshort = t(1)
	s = {t:'2d', x:x, y:y, d:d, e: energy, xl:xl, yl:'Zone Plate Z (um)  ch '+let, dl: fileshort}
		text = string(format= $
	  '(i4," x",i4," focus scan read from ",A,". E =",f7.2," eV. Dwell =",f5.2," ms.")', $
	   nx, ny, file, energy, dwell)
	print, text
	if keyword_Set(group) then Widget_CONTROL, Uprompt, Set_Value = text, /append
	return, s
	END

	'XANES': BEGIN
	; ------- select data channel
		t = ax_name(file)
		file = t(0) + t(1) + '.xsp'
		test = findfile(file)  & test = test(0)
		if test EQ '' then $
		  file = pickfile2(title = 'Select spectral file name', FILTER='*.xsp', /LPATH, DEFPATH=defpath)
		if file EQ '' then return, ''
		openr,lun,file, /get_lun			; re-open to get data
;		for i = 0, (count - npts - 1) do readf,lun,test            ; ONLY when header at top
; ------ now positioned at start of multi-column data
		test = ''
		energy = float(ScanDef.ScanDefinition.Paxis.points)
		npts = n_elements(energy)
		d = fltarr(npts)  & i =0
		ldata = fltarr(n_channels+1)
		Print, file, '       # points: ', strtrim(string(npts),2),'      # data channels: ', strtrim(string(n_channels),2)
		for i = 0, npts-1 do begin
			readf,lun, line
;					print, 'line ', i,': ', line
			reads, line, ldata
			energy(i) = ldata(0)
			d(i)  = ldata(n_data)
		endfor
		close, lun
;		plot, energy,d
; --------------- generate axis 1d structure -----
		type='1d'
		dwell_time = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		t = ax_name(file)
		fileshort = t(1)
		xl = string(FORMAT='("Energy(eV)    dwell = ",f7.2," ms")', dwell_time)
		tmp = {t:type, x:energy, d:d, dn:d, xl:xl, dl:fileshort}
		print, 'read ALS BL5.3.1 XANES file: ', file
		npts = n_elements(tmp.x)
		print, format='(i4," pts. ",f9.4," to ",f9.4," eV. Dwell= ",f7.2," ms.")', $
		    npts, tmp.x(0), tmp.x(npts-1), dwell_time
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
		return, tmp
	END

	'XANES Image Scan': BEGIN
		Print, 'Reading in ALS Beamline 5.3.1 Stack'
		text = 'ALS 5.3 stack read in: not yet developed'
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text
		s = 0
		return,s
	END

	'XANES Line Scan': BEGIN
		y = float(ScanDef.ScanDefinition.PAxis.points)
		x = float(ScanDef.ScanDefinition.QAxis.points)
		dwell = ScanDef.ImageScan.EnergyRegions[0].DwellTime
		t = ax_name(file)
		img_file = t(0) + t(1) + '_' + let +'.xim'
		d = float(text_read1(img_file))
		help, d
		xl = string(FORMAT='("x (um)   dwell = ",f5.2," ms")', $
     			 dwell)
		s = {t:'2d', x:x, y:y, d:d, e: 0, xl:xl, yl:'um  ch '+let, dl: fileshort}
		text = string(format= $
	  	    '(" Linescan read from ",A,". Emin =",f7.2," eV. Emax =",f7.2," eV.  Dwell =",f5.2," ms.")', $
	   		   file, x(0), x(n_elements(x)-1), dwell)
		print, text
		if keyword_Set(groupID) then Widget_CONTROL, Uprompt, Set_Value = text, /append
		return,s
	END
	; -------- bailout -------------------------
	ELSE: BEGIN
		print, 'Unknown scan type'
	ENDELSE
ENDCASE
return, scandef

nofile:
print, 'Open or read error on ', file
close,/all
return, s
end

