; Copyright (c) 1998-2008 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		READ_BESSY_STACK.PRO
;
;LAST CHANGED: ----------------------------------- 	13-Oct-08
;
; PURPOSE:
;	This function reads a sequence of images measured with Bessy STXM (Thieme)
; and converts the set to an  aXxis2000 *.ncb format
;
; CATEGORY:
;	data read routine. can execute outside of aXis2000
;
; CALLING SEQUENCE:
;	Result = READ_BESSY_STACK(directory=directory, filter=filter, normalize=normalize, silent=silent)
;
; INPUTS:
;
; KEYWORDS:
;	DIRECTORY:  name of directory
;	FILTER: extension (default for images is '*.pgm')
;	NORMALIZE: if set, modify intensities to standard ring current (value of normalize parameter
;	SILENT: if set, do not print tracking comments
;
; OUTPUTS:
;	No explicit outputs.
;
; COMMON BLOCKS:
; AXIS_COM	standard set of common blocks
; STACK_PROCESS_COM
; BSIF_COM
; volume_data, image_stack
;
; SIDE EFFECTS: none
;
; RESTRICTIONS: none
;
; PROCEDURE:
;
; MODIFICATION HISTORY:
; (13-Oct-08 aph) first version
;-

function read_bessy_stack, directory=directory, filter=filter, normalize=normalize, silent=silent
@axis_com
@bsif_com
@STACK_PROCESS_COM
COMMON volume_data, image_stack

on_error,2

check=0
if not keyword_set(filter) then filter='*.pgm'

if n_elements(directory) eq 0 then begin  ;popup file dialog box
   file=PICKFILE2(/Read, FILTER=filter, /LPATH, DEFPATH=defpath)
endif

if strlen(file) GT 0 THEN BEGIN
	name_bits = ax_name(file)	; sort the name into its parts
	path = name_bits(0)
	fileshort = name_bits(1)
    ext = name_bits(2)
 endif else begin
	axis_log, 'Directory ' + path +  ' is not defined, or does not exist'
endelse

if not keyword_set(silent) then axis_log, 'Reading Bessy STXM stack: ' + path

; =================
; construct a list of the files to read (all *.pgm in the directory)
cd, path
fnames=file_search(filter, count=a)
if a LE 0 then begin
	axis_log, 'no *.pgm files found'
	return, check
endif
n_imgs = n_elements(fnames)
filename_ev_msec_list = strarr(n_imgs)

if not keyword_set(normalize) then begin
	norm=300.  ; normalize to standard start-of-fill value (or top-up)
	t=dialog_message('Normalize to ring current ?',/question)
	if t EQ 'Yes' then begin
		if n_elements(axis_ID) EQ 0 then norm = get_num(prompt='standard ring current', val=norm) $
		       else norm = get_num(prompt='standard ring current', val=norm, group = axis_ID)
	endif
endif else norm = normalize
if not keyword_set(silent) then axis_log, 'Normalize to ' + string(norm) + ' mA'
; --------- read first image to get image size and other parameters
ev=fltarr(n_imgs)
image=read_bessy_stxm(file=fnames(0), /silent, normalize=norm)
i=0
t=ax_name(fnames(i))
filename_ev_msec_list(i) = t(1)+ string(image.e, format='(G13.6)')
ev(i)=image.e
n_cols = n_elements(image.x)
n_rows = n_elements(image.y)
x_start = image.x(0)
x_stop  = image.x(n_cols-1)
y_start = image.y(0)
y_stop  = image.y(n_rows-1)
xstep = (x_stop - x_start)/n_cols
ystep = (y_stop - y_start)/n_rows
if not keyword_set(silent) then axis_log, string(i) + '   ' + filename_ev_msec_list(i)
image_stack=fltarr(n_elements(image.x), n_elements(image.y), n_imgs)
image_stack(*,*,i) = image.d

; -------- now process the other images
for i=1, n_imgs-1 do begin
	image=read_bessy_stxm(file=fnames(i), /silent, normalize=norm)
	ev(i)=image.e
	image_stack(*,*,i) = image.d
	t=ax_name(fnames(i))
	filename_ev_msec_list(i) = t(1)+ string(image.e, format='(G13.6)')
	if not keyword_set(silent) then axis_log, string(i) + '   ' + filename_ev_msec_list(i)
endfor

; return, image_stack
; end

;  write image_stack then
stackname = pickfile2(/read, prompt='Stack name', filter='*.ncb')
tn=ax_name(stackname)
stackname=tn(0)+tn(1)+'.ncb'
stack_wb, stackname
t=ax_name(stackname)
if not keyword_set(silent) then axis_log, 'generated Bessy stxm stack ' + stackname
check=1
return, check
end