; Copyright (c) 1998-2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		FILENAME_IN_CAPITALS
;
;LAST CHANGED: ----------------------------------- 	22 Jul 2014
;
; PURPOSE:
;	This procedure rites out an image in netCDFformat
; It assumes that you have put the image and associated information
; into BSIF_COMMON variables:
;   image_data (the array which holds the image - possible several
;		planes deep for I, IO, CLOCK)
;   x_normal, y_normal, rotated, x_title, y_title
; ;
; CATEGORY:
;	Data I/O
;
; CALLING SEQUENCE:
;   write_ncdf, file, help=help
;	ZOOM [, FACT = Fact, /INTERP, XSIZE = Xs, YSIZE = Ys, /CONTINUOUS, $

; INPUTS:
; 	FILE	name of output file
;
; KEYWORDS:
;	HELP	display a help string
;
; OUTPUTS:
;	*.nc file written
;
; COMMON BLOCKS:
;	BSIF.COM	standard set of common blocks for nc files

; MODIFICATION HISTORY:
; (01-Jun-93 cjj) ; Based on Mark Rivers stuff
; (22-Jul-14 aph) AXIS standard header added
;-

pro write_ncdf, file, help=help
@bsif_com
on_error,2

if (keyword_set(help) or (n_elements(file) eq 0)) then begin
  print,'write_ncdf,filename'
  print,'  Writes image_data to a netCDF file.'
endif

; Make sure image_data has at least three dimensions
svec=size(image_data)
image_data_type=svec(svec(0)+1)
if (svec(0) eq 0) then begin
  n_cols = 1
  n_rows = 1
  n_data = 1
endif else if (svec(0) eq 1) then begin
  n_cols = svec(1)
  n_rows = 1
  n_data = 1
endif else if (svec(0) eq 2) then begin
  n_cols = svec(1)
  n_rows = svec(2)
  n_data = 1
endif else if (svec(0) eq 3) then begin
  n_cols = svec(1)
  n_rows = svec(2)
  n_data = svec(3)
endif else begin
  message,'image_data has '+strtrim(string(svec(0)),2)+ $
    ' rather than <=3 dimensions'
endelse
image_data=reform(image_data,n_cols,n_rows,n_data,/overwrite)

; Make sure data_title is a byte array
data_title=byte(data_title)
svec=size(data_title)
n_chars=svec(1)
temp_title=data_title
data_title=bytarr(n_chars,n_data)
for title_index=0,(n_data-1) do begin
  print,'title_index is '+string(title_index)
  data_title(0:(n_chars-1),title_index)= $
    temp_title(0:(n_chars-1),title_index)
endfor

; Make sure x_title means something
if (n_elements(x_title) eq 0) then begin
  x_title='X title'
endif
x_title=byte(x_title)

; Make sure y_title means something
if (n_elements(y_title) eq 0) then begin
  y_title='Y title'
endif
y_title=byte(y_title)

; Make sure the string image_title means something
if (n_elements(image_title) eq 0) then begin
  image_title='Image title'
endif
image_title=byte(image_title)

; Create output file
cdfid = ncdf_create(file, /CLOBBER)

; Create dimensions
col_id = ncdf_dimdef(cdfid, "n_cols", n_cols)
row_id = ncdf_dimdef(cdfid, "n_rows", n_rows)
data_id = ncdf_dimdef(cdfid, "n_data", n_data)
char_id = ncdf_dimdef(cdfid, "n_chars", n_chars)

; Get NetCDF data type
; data_type = nc_datatype(image_data)

; Create variables
if (image_data_type eq 1) then begin
  ; byte is /byte
  image_data_id = ncdf_vardef( cdfid, "image_data", $
			 [col_id, row_id, data_id], /byte )
endif else if (image_data_type eq 2) then begin
  ; short int is /short
  image_data_id = ncdf_vardef( cdfid, "image_data", $
			 [col_id, row_id, data_id], /short )
endif else if (image_data_type eq 3) then begin
  ; long int is /long
  image_data_id = ncdf_vardef( cdfid, "image_data", $
			 [col_id, row_id, data_id], /long )
endif else if (image_data_type eq 4) then begin
  ; float is /float
  image_data_id = ncdf_vardef( cdfid, "image_data", $
			 [col_id, row_id, data_id], /float )
endif else if (image_data_type eq 5) then begin
  ; double is /double
  image_data_id = ncdf_vardef( cdfid, "image_data", $
			 [col_id, row_id, data_id], /double )
endif else begin
  message,'Cannot handle data type '+strtrim(string(image_data_type),2)
endelse
data_title_id = ncdf_vardef( cdfid, "data_title", $
			 [char_id, data_id], /char )

; Define global attributes
ncdf_attput, cdfid, /GLOBAL, "title", image_title
bsif_netcdf_version='1.0'
ncdf_attput, cdfid, /GLOBAL, "bsif_netcdf_version", bsif_netcdf_version

; Save attributes for BSIF stuff
ncdf_attput, cdfid, image_data_id, "x_normal", x_normal
ncdf_attput, cdfid, image_data_id, "y_normal", y_normal
ncdf_attput, cdfid, image_data_id, "rotated", rotated
ncdf_attput, cdfid, image_data_id, "x_start", x_start
ncdf_attput, cdfid, image_data_id, "x_stop", x_stop
ncdf_attput, cdfid, image_data_id, "y_start", y_start
ncdf_attput, cdfid, image_data_id, "y_stop", y_stop
ncdf_attput, cdfid, image_data_id, "x_title", x_title
ncdf_attput, cdfid, image_data_id, "y_title", y_title

; End define mode
ncdf_control, cdfid, /ENDEF

; Write out data
ncdf_varput, cdfid, image_data_id, image_data
ncdf_varput, cdfid, data_title_id, data_title

; Close file
ncdf_close, cdfid

return
end


