; Copyright (c) 1995-2013 A.P. Hitchcock  All rights reserved
;+
;NAME:
;		XAS1.PRO
;
;LAST CHANGED: ----------------------------------- 	30-Aug-2013
;
; PURPOSE:
;	This function reads Beamline 7 STXM X-ray absorption spectral data (1995-1998)
;
; CATEGORY:
;	data read in
;
; CALLING SEQUENCE:
;	Result = xas1(file, plot=pl, DEFPATH=defpath, OPT=opt, _extra=e)
;
; INPUTS:
;	FILE	filename
;
; KEYWORDS:
;	PLOT	if set, display data
; default (no opt) is to read in E,col1,col2
; OPT = col1:   raw data - ASSUMES transmission detector in col(1); OSA in col(2)
; OPT = col2:  extract col(2)  - OSA (pre-98) or trans (after Dec-97; flipped here)
; OPT = col3;  extract col(3)  - TEY
; OPT = dark:  dark correct signals
; OPT = OSA;   use OSA signal for Io & convert to rel-O.D.  (no dark correct)
; OPT = OSA-dark; dark correct, then use OSA signal for Io & convert to rel-O.D.
;
; OUTPUTS:
;	No explicit outputs.   A new window is created if necessary
;
; COMMON BLOCKS: none
;
; MODIFICATION HISTORY:
; (09-Dec-97 aph)  multi-section files; single file to do all types of read-in of ALS STXM spectral files
; (15-feb-98 aph) cancel protect; read 0 dark_pt  files
; (04-mar-98 aph) still problems with 0 dark pts;
; (09-apr-98 aph) save img_shifts; correct spike; add col(3) = TEY
; (07-Aug-98 aph) indicate extension (? not done since col indicated already)
; (30-Aug-13 aph) AXIS standard header added
;-

 function xas1, file, plot=pl, DEFPATH=defpath, OPT=opt, _extra=e

on_error,2

if n_elements(file) eq 0 then begin  ;popup file dialog box
   fltr='*.xas'
   if keyword_set(nof) then fltr='*'
   file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath)
endif
if strlen(file) GT 0 THEN BEGIN
	file=strlowcase(file)                  ; assumes ALS 8.3 filename
	file=strmid(file,0,strlen(file)-4)
	dat=text_read1(file+'.dat')
	a=text_read1(file+'.xas')
	icst=a(0,*)
	b=a(1,*)
	c=a(2,*)

; AUTO-detect file type (aph - 6-dec-97)
test = size(dat)
if test(1) EQ 3 then begin
	nul=dat(0,1)	; (format A) pre-Nov-97 - 1-section file
	step=dat(1,0)
	dw = dat(2,1)/1e3  ; convert dwell in usec to msec
	samp=dat(2,0)
	ien=dat(0,0)
	b = b/dw  &  c = c/dw  ; normalize to dwell in msec
endif else begin		; post Nov-97 formats - multi-section files
; (format B)(Nov-20 to 1-Dec-97) - no dark channel info - deduce from file
    if dat(1,0) NE (test(2)-1) then begin  ;(format B)
        nseg = test(2)                 ; number of segments
        xtst = size(a)
        samp = total(dat(2,*))          ; total number of data points
        nul = (xtst(2) - samp)/2       ; number of dark points
        first = 0 & last = nseg - 1
    endif else begin
; (format C) (after 3-Dec-97) - 1st line of dat has dark channel info
; ALSO somewhere (perhaps Feb-98 ?) OSA and TRANS signals swapped
	    t=b & b=c & c=t                ; so swap them back !!
		if opt EQ 'col3' then begin    ; only attempt readin if requested
		   t = size(a)
		   if t(1) GE 4 then begin    ; bail if no col(3)
			   d=a(3,*)
			   print, 'read col3 from ', file, '  : ', $
			   strcompress(string(t(1))), ' cols ', strcompress(string(t(2))), ' points'
			   print, 'pt 20 = ',d(20)
		   endif else goto, no_tey
		endif
		nseg = dat(1,0)               ;(format C)
		samp  = total(dat(2,1:*))
    	nul = dat(0,0)
    	first = 1 & last = nseg
    endelse
; common to both post_Nov-97 formats
    nb = fix(nul)
    for i = first,last do begin    ; correct for different dwell for each section
       dw = dat(3,i)/1e3
       nf = fix(nb + dat(2,i))     ; 7-apr-98 - increased nf by 1 to avoud spike
;       print,'section ',i,' dwell (ms) = ',dw, ' npts = ',nf-nb
       b(nb:nf) = b(nb:nf)/dw
       c(nb:nf) = c(nb:nf)/dw
       if opt EQ 'col3' then  d(nb:nf) = d(nb:nf)/dw
       nb = nf + 1
    endfor

; correct dark counts for dwell (NB can have error here for format B)
if nul GT 1 then begin  ; don't if no dark counts
	 dw = dat(3,0)/1e3
	 b(0:nul-1) = b(0:nul-1)/dw
	 b(nul+samp+1:*)=b(nul+samp+1:*)/dw              ; 7-apr-98 increment
	 c(0:nul-1) = c(0:nul-1)/dw                      ; to avoid spike
	 c(nul+samp+1:*) = c(nul+samp+1:*)/dw
	 if opt EQ 'col3' then  d(nul+samp+1:*) = d(nul+samp+1:*)/dw
	; print, '#dark= ',fix(nul),' dwell dark (ms) ',dw,'   #data= ', samp
endif
endelse

file=strmid(file,strlen(file)-8,strlen(file))
ics = icst
ytit = 'DET and OSA(rescaled)'
;if NOT keyword_set(opt) then OPT = 'raw' ; default

if OPT EQ 'dark' then begin
	bd=b(nul:nul+samp-1)
	cd=c(nul:nul+samp-1)
	bi=b(0:nul-1)
	bf=b(nul+samp:*)
	ci=c(0:nul-1)
	cf=c(nul+samp:*)
	ics=icst(nul:nul+samp-1)

	m1i=median(bi)
	m1f=median(bf)
	n1i=median(ci)
	n1f=median(cf)

	cia1=findgen(samp)*(n1f(0)-n1i(0))/samp+n1i(0)
	bia1=findgen(samp)*(m1f(0)-m1i(0))/samp+m1i(0)
	b=bd-bia1
	c=cd-cia1
	ytit = 'DET and OSA(rescaled) [dark current subtracted]'
endif
if OPT EQ 'OSA' then begin
   	ics=icst(nul:nul+samp-1)
	b1=b(nul:nul+samp-1)
	c1=c(nul:nul+samp-1)
   	b = -alog(b1/c1)
   	c = c1
   	ytit = ' ln (DET/OSA) [no dark correction]'
endif

if OPT EQ 'OSA-dark' then begin
  	bd=b(nul:nul+samp-1)
	cd=c(nul:nul+samp-1)
	bi=b(0:nul-1)
	bf=b(nul+samp:*)
	ci=c(0:nul-1)
	cf=c(nul+samp:*)
	ics=icst(nul:nul+samp-1)

	m1i=median(bi)
	m1f=median(bf)
	n1i=median(ci)
	n1f=median(cf)

	cia1=findgen(samp)*(n1f(0)-n1i(0))/samp+n1i(0)
	bia1=findgen(samp)*(m1f(0)-m1i(0))/samp+m1i(0)
	bd=bd-bia1
	cd=cd-cia1
	if min(cd) LT 0 then cd = cd - min(cd) + 1
	uf=(bd/cd)
	b=-alog(uf)
	c = cd
	ytit = ' ln (DET/OSA) [dark corrected]'
endif

if opt EQ 'col2' then b = c
if opt EQ 'col3' then begin
	 b = d
	 print, 'pt 20 = ',b(20)
endif

s={t:'1d', d:b, x:ics, xl:'Photon Energy (eV)',dn:c, dl:file}

if keyword_set(pl)then begin
   	splot,s,ytitle=ytit,_extra=e,charsize=1
  	if OPT EQ 'raw' then oplot,s.x,s.dn/max(s.dn)*max(s.d),color=129
  	if OPT EQ 'dark' then oplot,s.x,s.dn/max(s.dn)*max(s.d),color=129
endif

return, s
endif
print, 'Filename of zero length (Cancel)'
s = 0
return, s
no_tey:
print, 'No col(3) signal in file', file
s = 0
return, s
end

