; Copyright (c) 1998-2000 A.P. Hitchcock  All rights reserved 
;+ 
;NAME: 
;	NCDF2ALS 
; 
;LAST CHANGED: ----------------------------------- 20-May-00 
; 
; PURPOSE: 
;	This procedure converts one or a series of NSLS image files (*.nc) 
; into ALS ascii format (*.img) with a user definable scale factor 
; 
; CATEGORY: 
;	AXIS: image analysis 
; 
; CALLING SEQUENCE: 
;	NCDF2ALS, file 
; 
; INPUTS: 
;	FILE - netCDF format (*.nc) to be converted to ALS ascii format 
; 
; KEYWORD PARAMETERS: 
;	LIST 	- name of file with list of image files (*.sl) 
;   OUTLIST - name of a file to write list of netCDF files (*.lst) 
;	ONE 	- convert only a single file 
;	SCALE 	- scale factor (nb ALS format is integer) 
; 
; OUTPUTS: 
;  FILE.img - image written to a ALS ascii format binary file (*.img, *.dat) 
; 
; COMMON BLOCKS: 
;   @AXIS_COM - set of common blocks for AXIS 
;   @BSIF_COM - common block for netCDF 
; 
; RESTRICTIONS: 
;	caveat emptor 
; 
; PROCEDURE: 
;	NCDF2ALS uses READ_BNL to put the image and associated information 
; into BSIF_COMMON variables: 
;	image_data (the array which holds the image - possibly several 
;	planes deep for I, IO, CLOCK) 
;	then uses SAV_ALS to write the ALS data files 
; 
; MODIFICATION HISTORY: 
; (20-may-00 aph) adapted from als3ncdf 
;- 
 
pro ncdf2als, file , list=list, bin=bin, outlist = outlist, one = one, scale = scale 
 
@axis_com
@bsif_com
on_error,2 
 
if keyword_set(list) then begin 
   stack_readlist, list, filename_list 
endif else begin 
	if n_elements(file) eq 0 then begin  ;popup file dialog box 
	   if keyword_set(one) then fltr='*.nc' else fltr='*.sl' 
	   file=PICKFILE2(/Read, FILTER=fltr, /LPATH, DEFPATH=defpath) 
	endif 
	filename_list=file 
endelse 
 
n_files = n_elements(filename_list) 
if keyword_set(delE) then del_E = float(delE) else del_E=0. 
 
; -------- use GET_PATH to establish correct file names 
; -------- check if file exists - this handles subdirectory changes 
file1 = filename_list(0) 
New_Path = get_path(file1, list = list) 
if New_Path EQ '' then return	; let user cancel 
for i = 0, n_files-1 do begin 
	t = ax_name(filename_list(i)) 
	filename_list(i) = New_Path + t(1) + '.' + t(2) 
endfor 
 
; ------ open *.sl file for stack_list ----------------- 
if keyword_set(outlist) then begin 
; -------- force outlist to have .sl extension ----------------- 
	t = ax_name(outlist) 
	outlist = t(0) + t(1) + '.lst' 
	openw,unitw, outlist,/get_lun 
; ---- NB *.lst files from ALS stacks do not have a path at top --- 
;	printf, unitw, strcompress(New_Path)		; write path at top of file 
endif 
 
; -------------- check back for user to see if filename list is valid--------- 
	t = ax_name(filename_list(0)) 
;	print, 'First file to process is ', t(0) + t(1) + '.' + t(2) 
 
FOR i_file = 0,n_files-1 DO BEGIN 
	WIDGET_CONTROL, /hourglass 
	file = filename_list(i_file) 
	IF strlen(file) GT 0 THEN BEGIN 
;	print,' Procesing file ', fix(i_file), '. . . ', file 
		t = ax_name(file) 
		ext = t(2) 
		if ext NE 'nc' then goto, skip 
		fileshort = t(1) 
		Data_path = t(0) 
		file = t(0) + t(1) + '.' + t(2) 
; READ IN DATA 
		tmp = read_bnl(file) 
;		help, tmp,/structure 
		energy = 12398.0/sd.wavelength + del_E(0) 
		sd.wavelength = 12398.0/energy 
;		print, Format = '(A, "  E=",F7.3, "(eV).  Dwell=", f4.2 ,"(ms)")', $ 
;			file, energy, sd.dwell_time 
		data_title = byte(fileshort) 
		x_title = 'X' 
		y_title = 'Y' 
		image_title = byte(fileshort) 
		sd.clock_frequency = 1. 
; WRITE-OUT DATA 
		file = t(0) + t(1) + '.img'  ; setup correct name for output in nc format 
		if keyword_set(scale) then 	test = sav_als(tmp, file=file, scale=scale) else $ 
			test = sav_als(tmp, file=file) 
		if keyword_set(outlist) then printf,unitw, strcompress(t(1) + '.img') 
	ENDIF 
	skip: 
ENDFOR 
 
if keyword_set(outlist) then begin 
	close, unitw 
	free_lun, unitw 
	print, 'wrote list file: ', outlist 
	WIDGET_CONTROL, Uprompt, Bad_ID = badID, $ 
	    SET_VALUE='file list saved in ' + strcompress(outlist) 
endif 
close,/all		; useful to ensure logical units are available ! 
 
if NOT keyword_set(list) then begin	; plot if only 1 file converted 
	t = Widget_info(/active)	; BUT only if AXIS is running 
	if t NE 0 then begin 
		tmp = read_als(file) 
		Label(CurBuf) = tmp.dl 
		HANDLE_VALUE, Data(CurBuf), tmp, /set 
		PlotBuf, CurBuf 
	endif 
endif 
END 
