; Copyright (c) 1998-2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	TIFRD_IM.PRO
;
;LAST CHANGED: ----------------------------------- 21-Jul-14
;
; PURPOSE:
;	This procedure reads a tif image file (*.tif) using RD_PEEM  and converts it to
; NetCDF format. This is used to process *.tif to *.nc files for use in stack_analyze
;
; CATEGORY:
;	Image format conversion.
;
; CALLING SEQUENCE:
; for procedures:
;	TIFRD_IM, [file, bin=bin, Energy=energy, white=white, CCD_bgnd=CCD_bgnd, $
;        smth = smth, bits12 = bits12, region = region, scale=scale, $
;		 help=help, _extra=e]
;
; INPUTS:
;	FILE 	name of file to convert (user prompted if not supplied)
;
; KEYWORDS:
;	BIN		paramater for binning
;	BITS12 	if set to 12 this uses the 12-bit read routine in RD_PEEM
;	ENERGY	sets energy of PEEM image
;	CCD_bgnd CCD background image or average value that is subtracted
;	SCALE 	pixel size in microns
;	SMTH 	if set, applies a 3-point median smooth
;	WHITE 	if set, divides the image by an image in file white
;           (response of CCD to uniform illumination)
;	AOI 	area of interest, defined by box, [x0,x1, y0, y1] acquired by PEEM-2
;			used to extract su-image from dark and white files
;	REGION 	a 4-number vector [x_start, x_stop, y_start, y_stop]
;			defining the region of the image to be selected
;			(if scale set, in real space units. Otherwise, in pixels)
;	NOFILTER if set, then pickfile dialog does not have filter set to '*.tif'
;	HELP	print information about this routine
;	_EXTRA	any other parameters to be passed on to other procedures
;
; COMMON
;	AXIS_COM general common blocks for AXIS
;	BSIF_COM - common for netCDF format
;
; OUTPUTS: data is set-up in image_data and with bsif_common variables set
;	 ready to write *.nc if desired (as is done from tif2ncdf)
;
; MODIFICATION HISTORY:
; (13-May-99 aph) adapted from ax_rd_im; then to using rd_peem
; (08-jun-99 aph) add group to get_num call
; (17-jul99 aph) CCD_bgnd introduced
; (24-oct-00 aph) allow 12-bit or 16-bit ALS PEEM readin
; (06-may-01 aph) AXIS standard header added; gain for dynamic range deleted;
;				format of image_data switched to float
; (12-jul-01 aph) correct for dark on read-in if aio_dark supplied
; (31-jul-01 aph) include /axis keyword in call to RD_PEEM (assumes only run from axis)
; (21-Jul-14 aph) corrected header
;-

pro tifrd_im, file, bin=bin, Energy=energy, white=white, CCD_bgnd=CCD_bgnd, $
         smth = smth, bits12 = bits12, region = region, scale=scale, $
         aoi_box = aoi_box, aoi_data = aoi_data, aoi_gain=aoi_gain, aoi_dark=aoi_dark, $
         help=help, _extra=e
on_error,2
@axis_com
@bsif_com

IF ((n_params() eq 0) or keyword_set(help)) THEN BEGIN
    print,'TIFRDIM:  Reads TIF format files for conversion to netCDF format'
    print,'tifrd_im, file bin=bin, Energy=energy, white=white, CCD_bgnd=CCD_bgnd, $'
    print,'    smth = smth, bits12 = bits12, region = region, scale=scale, $'
	print,'		 help=help, _extra=e'
    return
ENDIF

i_dat_file = 1
aoi_dark = 0
aoi_gain = 0
aoi_data = 0
tmp = rd_peem(file, Energy = energy, bits12 = bits12, white = white, CCD_bgnd=CCD_bgnd, $
            smth = smth, region=region, scale=scale, _extra=e, /axis,  $	; ASSUMES ONLY CALLED FROM AXIS
            aoi_box = aoi_box, aoi_data = aoi_data, aoi_gain=aoi_gain, aoi_dark=aoi_dark)
x_start = min(tmp.x)    &   x_step = tmp.x(1) - tmp.x(0)
y_start = min(tmp.y)    &   y_step = tmp.y(1) - tmp.y(0)
n_cols = n_elements(tmp.x)       &   n_rows = n_elements(tmp.y)
x_stop = x_start + x_step*n_cols
y_stop = y_start + y_step*n_rows
;print, filelist(i_dat_file)
print,'      Image size: '+strtrim(string(n_cols),2)+' x '+strtrim(string(n_rows),2)
dwell = 1000.
sd.dwell_time = dwell/1000.    ; report dwell in msec
if (energy NE 0)then begin
;	print, format='("Energy (eV) = ",f7.3,". Dwell (ms) = ",f6.2)', energy, sd.dwell_time
	sd.wavelength=12398./energy    ; save photon energy as wavelength
endif else sd.wavelength=43.
; establish variable set needed for nsls SD
x_title = 'X'
y_title = 'Y'
x_normal=1
y_normal=1
rotated=0
image_title = byte(file)
tab_char = string(byte(9))
i_data = 0
i_col = 0
i_row = 0
n_data = 2
image_data = fltarr(n_cols,n_rows,n_data)
image_data(*,*,0) = tmp.d

IF keyword_set(bin) then begin
   	IF bin GT 1 THEN BEGIN
      if (float(n_cols)/float(bin))-fix(n_cols/bin) GT 0 $
        OR (float(n_rows)/float(bin))-fix(n_rows/bin) then begin
        nct =fix(n_cols/bin)*bin  &  nrt = fix(n_rows/bin)*bin
        xtra_c = n_cols - nct + 1
        xtra_r = n_rows - nrt + 1
;        print, 'truncate image to ', fix(nct), fix(nrt)
        x_stop = x_stop*float(nct)/float(n_cols)  ; correct axis length for truncation
        y_stop = y_stop*float(nrt)/float(n_rows)
        tmp = image_data                      ;force size to integer mutiple of bin
        image_data= fltarr(nct,nrt,n_data)
		image_data=tmp(0:n_cols-xtra_c,0:n_rows-xtra_r,0:n_data-1)
      endif
      n_cols=fix(n_cols/bin)
      n_rows=fix(n_rows/bin)
;      print, 'bin image ', fix(bin), ' times,  to ',n_cols,'  x', n_rows
      image_data=rebin(image_data,n_cols,n_rows,n_data)
    ENDIF
  ENDIF

; ------------ this gave problem for background correction - why was it here ? (aph 6-may-01)
;IF (min(image_data) LT 0) THEN BEGIN
;    image_data = long(image_data)
;    negatives = where(image_data LT 0)
;    image_data(negatives) = long(65536)+long(image_data(negatives))
;ENDIF

image_data(*,*,1)= 1
sd.clock_frequency = 1.
; free_lun,lun
return

END


