; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_SINGLE_STACK.PRO
;;
;LAST CHANGED: ----------------------------------- 	28 Mar 2024 (aph)

; PURPOSE:
;	This function reads a set of images using the read_nanomax_image routine to read NanoMAX h5 files
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	RESULT = READ_NANOMAX_PTYCHO_STACK(path=path, verbose=verbose, group = group)
;
; INPUTS: none
;
; KEYWORDS:
;	PARTH
;	VERBOSE
;	GROUP
;
; OUTPUTS:  stack as (*.dat, *.ncb)
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (28-Mar-24 aph) first version

;-

FUNCTION  read_nanomax_single_stack, path=path, silent=silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0

; ------------ select NanoMAX  *.h5 file with stack

file = dialog_pickfile(title='Select NanoMAX single stack  *.h5 files',filter=['*.h5'], path=DefPath, get_path=folder_path, /multiple_files)
DefPath=folder_path
cd, DefPath
print, "new default path is ", DefPath

a = h5_parse(file, /read_data)

; ---------- get photon energies
energy = a.info.energies._data

print, 'energies ', energy

n_e = n_elements(energy)

; ---------- get sequential scan numbers
scan_num= a.info.scans._data
;print, scan_num

; ---- First format - all data ln(signal/(ion_chamber))
; image_stack = a.maps.stxm.data._data ; exp(a.maps.stxm.data._data);  need to exp to undo the normalization to the ion chamber, Maik did
; help, image_stack

image_stack =  a.maps.raw_stxm.data_it_Eiger._data/1e8

image_stack_io=  a.maps.raw_stxm.data_I0._data/1e-8

image_stack = image_stack / image_stack_io

tmp = size(image_stack)

nx = tmp(1)
ny = tmp(2)

; ------- generate x, y axes  &  store in bsif_com parameters
ev = energy

pix_size_array = a.maps.raw_stxm._psizes._data
pix_size=mean(pix_size_array*1e9)
x = findgen(nx)
x = pix_size*x
y = findgen(ny)
y = pix_size*y

x_start = x(0)
x_stop  = x(n_elements(x)-1)
x_step = (x_stop - x_start) /(nx-1)

y_start = y(0)
y_stop  = y(n_elements(y)-1)
y_step = (y_stop - y_start) /(ny-1)

n_rows = nx
n_cols = ny

filename_ev_msec_list = strarr(n_e)

; ---------- loop through to build tag line of stack and flip thhe images (Y ==> -Y)
for i = 0, n_e-1 do begin
	d = image_stack(*,*,i)
	d = rotate(d, 7)		; exp to undo the normalization
	image_stack(*,*,i) = exp(d)
	filename_ev_msec_list(i) = 'scan ' + strtrim(string(scan_num(i)),2) + '  ' + string(energy(i))
	axis_log, filename_ev_msec_list(i)
endfor

if n_elements(stack_file) LE 0 then stack_file = ' '
stack_file = get_text(prompt = 'name of stack ', val = stack_file, group = group)

stack_file = DefPath + stack_file
stack_wb, stack_file  ;, /real

return, stack_file

end