; Copyright (c) 1998-2024 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	READ_NANOMAX_SPECTRUM.PRO
;;
;LAST CHANGED: ----------------------------------- 	04 Apr 2024 (aph)

; PURPOSE:
;	This function reads an image from a single NanoMAX h5 file
;
; CATEGORY:
;	    Runs stand alone or inside aXis2000.
;
; CALLING SEQUENCE:
;	READ_NANOMAX_IMAGE = READ_NANOMAX_IMAGE(file = file, path=path, silent = silent, verbose=verbose, group = group)
;
; INPUTS:
;	FILE	filename.
;
; KEYWORDS:
;	KEY1:  none
;
; OUTPUTS:
;
;
; COMMON BLOCKS:
; AXIS_COM    standard set of common blocks
; stack_process_com
; COMMON volume_data, image_stack
; bsif_com


; MODIFICATION HISTORY:
; (27-Mar-24 aph) first version
; (04-Apr-24 aph) add save of last info

;-


FUNCTION  read_nanomax_spectrum, file = file, path=path, silent = silent, verbose=verbose, group = group
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

null = 0

; ------------ select NanoMAX  *.h5 file(s)
if n_elements(last_spectrum) EQ 0 then last_spectrum = ' '
if NOT keyword_set(file) then begin
    file = dialog_pickfile(title='Select NanoMAX SPECTRUM *.h5 file',file=last_spectrum, filter=['*.h5'], path=DefPath, get_path=folder_path)
	DefPath=folder_path    ; --- force path for writing to be the source folder
endif
if n_elements(file) EQ 0 then return, null

t = ax_name(file)
DefPath=t(0)
if NOT keyword_set(silent) then print, 'default path set to  ', DefPath

; ----------- single image read-in -------
if strlen(file) EQ 0 then RETURN, null
if NOT keyword_set(silent) then print, 'NanoMAX_spectrum_file = ' ,file
t =ax_name(file)
fileshort = t(1)



a = h5_parse(file, /read_data)

; --------- get dimensions of image from the header string
; ----------- TYPICAL string      mesh energy 8969 9029 120 1.0
hdr =  a.entry.description._data
h =strsplit(hdr,/extract)
;' print, 'HEADER   ', h
type = h(0)
emin = float(h(2))
emax = float(h(3))
n_e =  fix(h(4))+1

dwell = float(h(5))
; -- verify
if NOT keyword_set(silent) then print, 'Opening file ', file
if NOT keyword_set(silent) then print, " type , emin , emax, n_e, dwell(s)     ", $
            type , emin , emax, n_e, dwell


;energy = a.entry.snapshots.pre_scan.energy._data
;x_start = xmin
;x_stop  = xmax
;x_step = (xmax - xmin) /(nx-1)
;x = findgen(nx)*x_step + xmin
;y_start = ymin
;y_stop  = ymax
;y_step = (ymax - ymin) /(ny-1)
;y = findgen(ny)*y_step + ymin
;n_rows = nx
;n_cols = ny
;xl = '  ' +  string(energy) + ' eV.   dwell = ' +  string(dwell) + ' s'
;yl = ' '

energy = findgen(n_e)
e_step = (emax - emin) / n_e
energy = emin + energy*e_step

d =  a.entry.measurement.alba2._3._data			; diode after sample
dn =  a.entry.measurement.alba2._1._data		; upstream ion chamber

; 1-d file formst {t: d:, x:, xl:, dl:}

xl = 'Energy (eV)  dwell  (s) ' + string(dwell)
spectrum = {t:'1d', d:d, x:energy, xl:xl, dn:dn, dl:fileshort}


return, spectrum

end