; Copyright (c) 1998-2025 A.P. Hitchcock  All rights reserved
;+
;NAME:
;	AX_NEXUS_VIEWER
;
;LAST CHANGED: -----------------------------------  29-Sep-2025  (aph) from 31-Dec-2024 (aph)
;
;PURPOSE:
;	This set of procedures is a widget to view all Nexus format (hdf5) files in a folder
; with option to keep or to move the file into a sub-folder named "discard")
; if the folder "discard" does not exist it is created
;
;CATEGORY:
;	AXIS: utility, runs independent of aXis2000
;
; AX_NEXUS_VIEWER_EVENT ROUTINES:
; TEXT_SDF_FNAME		modify filename; when changed, execute BUT_SDF_FNAME_BROWSE to get parameters
; BUT_SDF_FNAME_BROWSE 	browse for new filename; if found, load new parameters
; ANV_BUT_DISMISS		dismiss the widget

;
;COMMON BLOCKS:
; @AXIS_COM     standard set of common for aXis2000
;	@NEXUS_COM		standard set of common for NEXUS
;
;MODIFICATION HISTORY:
; (08-May-16 aph) first version
; (10-MAy-16 aph)  add overwrite to file_move
; (01-Oct-18 aph) fix problem of not allowed to move to discard folder (was read_nexus not closing)
; (03-Jul-19 aph) add type indicator
; (27-Dec-24 aph) use preview_flag (stored in nexus_com) to force 1d, or 2d data to be displayed in the viewer
;		get transfer to \discard folder working  (NB weird "files-re-appear problem" , possibly since syncthing?
; (29-sep-25 aph) ensure the proview is active so can see data in NeXuS viewer
;-

;-----------------------------------------------------------------

pro ANV_SLIDER_select, Event
@axis_com
@nexus_com
on_error,2
	id = widget_info(Event.top, FIND_BY_UNAME = 'ANV_SLIDER_select')
	WIDGET_CONTROL,id, get_value = text
	anv_current_file = fix(text)
	if anv_current_file GE n_elements(anv_filelist) then $
		anv_current_file =  n_elements(anv_filelist)-1
	print, 'Current file # ', anv_current_file, '    ', anv_filelist(anv_current_file)
	t = ax_name(anv_filelist(anv_current_file))
	id = widget_info(Event.top, FIND_BY_UNAME = 'ANV_TEXT_filename')
	WIDGET_CONTROL,id, Set_VALUE = t(1)
	ANV_UPDATE
end


pro ANV_UPDATE
@axis_com
@nexus_com
on_error,2

	widget_control, /hourglass
	if anv_current_file GE n_Elements(anv_Filelist) then begin
		axis_log, 'No more NeXus_hdf5 files in current list'
		return
	endif
	anv_file = anv_Filelist(anv_current_file)
	print, 'Reading file: ', anv_file
	nexus_head = read_nexus(anv_file, /header_only, /silent)
	tmp = read_nexus(anv_file, channel= 0, region = 0, xy_correct=xy_correct_flag, one_image =0) ;, /no_save)
; ----- update filename
	t = ax_name(anv_filelist(anv_current_file))
	id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_TEXT_filename')
	WIDGET_CONTROL,id, Set_VALUE = t(1)

; ----- display type of file -----------
	id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_TEXT_type')
	WIDGET_CONTROL,id, Set_VALUE = nexus_head.Type

	if nexus_head.Type EQ 'sample image' then begin
		; ------ indicate E if possible
		id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_TEXT_energy')
		WIDGET_CONTROL,id, set_value = string(tmp.e)
		; ----- indicate dwell (ms)
		id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_TEXT_dwell')
		WIDGET_CONTROL,id, set_value = string(nexus_dwell, format='(F7.2)')
	endif
;	------ plot the data
	wset, ANV_NexusImg
	tmp.dl = ''
	make_square,tmp
	if tmp.t EQ '2d' then splot2d, tmp
	if tmp.t EQ '1d' then plot, tmp.x, tmp.d
end

;-----------------------------------------------------------------

pro ANV_BUT_KEEP, Event
@axis_com
@nexus_com
on_error,2

anv_file = anv_filelist(anv_current_file)
t = ax_name(anv_file)
axis_log, 'kept '+ t(1)
anv_current_file = anv_current_file +1
if anv_current_file GE  n_elements(anv_filelist) then anv_current_file = 0
; ---- change the maximum of the slider
	id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_SLIDER_select')
	WIDGET_CONTROL,id, set_slider_max = anv_nfiles
	ANV_UPDATE

end

;-----------------------------------------------------------------

pro ANV_BUT_DISCARD, Event
@axis_com
@nexus_com
on_error,2

	anv_file = anv_filelist(anv_current_file)


; ----- move the file  (NB used file_move but it seems read_nexus leaves that file open
;
; ----- When tried to use ax_nexus_viewer to aort data from Soleil Oct 2024, the
;       file_move did not work due to "Accessis denied' error
;
;;	Moving OSA Focus_2019-05-23_001  to  Y:\data-XRM\stxm-soleil\2019\19-05\05-23\discard
;Unable to rename file: Source  File: Y:\data-XRM\stxm-soleil\2019\19-05\05-23\OSA Focus_2019-05-23_001.hdf5,
;                  Destination  File: Y:\data-XRM\stxm-soleil\2019\19-05\05-23\discard\OSA
;Focus_2019-05-23_001.hdf5
;Access is denied.
;Error occurred at: AX_NEXUS_VIEWER_EVENT  153 C:\aXis2000\ax_nexus_viewer.pro

; make discard file if it doeds not exist
	t = ax_name(anv_file)
	discard =  anv_folder + 'discard'
	test = file_test(discard, /directory)		; check if discard exists
	if test EQ 0 then begin
		file_mkdir,discard
		print, 'discard folder crerated ', discard
	endif


; 12-Dec-2024 ==> use file_copy and file_delete command  ******
	print, 'Moving ', t(1), '  to  Discard folder'
;	file_move, anv_file, anv_folder + 'discard', /overwrite, /require_directory; ; 12-Dec-2024 **********
	file_copy, anv_file, anv_folder + 'discard', /overwrite
	file_delete, anv_file, /allow_nonexistent

; ---------- return if it is last file in the list
	if anv_current_file EQ n_elements(anv_filelist)-1 then begin
		axis_log, 'No more NeXus_hdf5 files in current list'
		ANV_BUT_DISMISS, Event
		return
	endif
; ---- modify anv_filelist to remove that name
	anv_nfiles = n_elements(anv_filelist)
;	print, '# of files in list before discard = ', 	anv_nfiles
	for i = anv_current_file, anv_nfiles-2 do  anv_filelist(i) = anv_filelist(i+1)
	anv_nfiles = anv_nfiles - 1
	anv_filelist = anv_filelist[0:anv_nfiles-1]
;	print, '# of files in list after discard = ', 	anv_nfiles
; ---- change the maximum of the slider
	id = widget_info(AX_NEXUS_VIEWER_ID, FIND_BY_UNAME = 'ANV_SLIDER_select')
	WIDGET_CONTROL,id, set_slider_max = anv_nfiles
	ANV_UPDATE
	print, t(1), '  moved to the discard folder'

end


;-----------------------------------------------------------------

pro ANV_BUT_DISMISS, Event
@axis_com
@nexus_com
on_error,2

WIDGET_CONTROL, Event.Top, /DESTROY
end


pro AX_NEXUS_VIEWER_event, Event
@axis_com
@nexus_com
on_error,2

  case Event.id of

    Widget_Info(Event.top, FIND_BY_UNAME='ANV_SLIDER_select'): begin
        ANV_SLIDER_select, Event
    end
    Widget_Info(Event.top, FIND_BY_UNAME='ANV_BUT_KEEP'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        ANV_BUT_KEEP, Event
    end
	Widget_Info(Event.top, FIND_BY_UNAME='ANV_BUT_DISMISS'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        ANV_BUT_DISMISS, Event
    end
    Widget_Info(Event.top, FIND_BY_UNAME='ANV_BUT_DISCARD'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        ANV_BUT_DISCARD, Event
    end
	Widget_Info(Event.top, FIND_BY_UNAME='ANV_TEXT_energy'): begin
      if( Tag_Names(Event, /STRUCTURE_NAME) eq 'WIDGET_BUTTON' )then $
        ANV_TEXT_energy, Event
    end
    else: print, 'AX_NEXUS_VIEWER - unknown button clicked'
  endcase

end

pro BASE_AX_NEXUS_VIEWER, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@nexus_com
on_error,2

; ------- ask user to identify folder and generate list of all *.hdf5 files
if n_elements(anv_folder) GT 0 then begin
	; should really check if anv_folder exists
	anv_filelist = dialog_pickfile(get_path=folder, title='Nexus file viewer: select files to process', $
	     /multiple_files, filter = '*.hdf5', path = anv_folder)
endif else  begin
	anv_filelist = dialog_pickfile(get_path=folder, title='Nexus file viewer: select files to process', $
	    /multiple_files, filter = '*.hdf5')
endelse

print, 'AX_NEXUS_VIEWER processing folder = ', folder
anv_folder = folder		; save for future access
anv_current_file = 0
anv_n_files = n_elements(anv_filelist)

; ------- kluge for windows list of files - last 1 is the first file !
if !version.os_family EQ 'Windows' then begin
    t=anv_filelist(0)
    for i = 0, anv_n_files-2 do  anv_filelist(i) = anv_filelist(i+1)
    anv_filelist(anv_n_files-1) = t
endif

print, anv_filelist

 BASE_ANV = Widget_Base( GROUP_LEADER=wGroup, UNAME='BASE_ANV '  $
      ,XOFFSET=5 ,YOFFSET=5 $
      ,TITLE='NEXUS file viewer' ,SPACE=3 ,XPAD=3 ,YPAD=3, column = 2)

  ANV_DRAW = Widget_Draw(BASE_ANV, UNAME='ANV_DRAW' ,XOFFSET=28  $
      ,YOFFSET=15 ,SCR_XSIZE=360 ,SCR_YSIZE=340)

col1 = widget_base(BASE_ANV,XOFFSET=400)

; --- use common length for folder and filenames
  anv_text_length = 50
  ANV_LABEL_folder = Widget_Label(col1, UNAME='ANV_LABEL_folder'  $
      , YOFFSET=1, /ALIGN_LEFT ,VALUE='Folder')
  ANV_TEXT_folder = Widget_Text(col1, UNAME='ANV_TEXT_folder'  $
      , YOFFSET=20,XSIZE=anv_text_length, YSIZE=1)

  ANV_LABEL_filename = Widget_Label(col1,  $
      UNAME='WID_LABEL_filename' ,YOFFSET=51  $
      , /ALIGN_LEFT ,VALUE='Filename')
  ANV_TEXT_filename = Widget_Text(col1,  $
      UNAME='ANV_TEXT_filename' ,YOFFSET=74  $
      , XSIZE=anv_text_length ,YSIZE=1)

  ANV_LABEL_type = Widget_Label(col1,  $
      UNAME='WID_LABEL_type' ,YOFFSET=104  $
      , /ALIGN_LEFT ,VALUE='Type')
  ANV_TEXT_type = Widget_Text(col1,  $
      UNAME='ANV_TEXT_type' ,XOFFSET = 75, YOFFSET=104, XSIZE=32)

   ANV_LABEL_Energy = Widget_Label(col1,  $
      UNAME='ANV_LABEL_Energy' , YOFFSET=137  $
      ,/ALIGN_LEFT ,VALUE='Energy')
   ANV_TEXT_energy = Widget_Text(col1, UNAME='ANV_TEXT_energy'  $
      ,XOFFSET = 75, YOFFSET=137, XSIZE=10, YSIZE=1)

    ANV_LABEL_Dwell = Widget_Label(col1,  $
      UNAME='ANV_LABEL_Dwell' , YOFFSET=175  $
      ,/ALIGN_LEFT ,VALUE='Dwell (ms)')
    ANV_TEXT_dwell = Widget_Text(col1, UNAME='ANV_TEXT_dwell'  $
      , XOFFSET = 75, YOFFSET=170, XSIZE=10, YSIZE=1)

  ANV_SLIDER_select = Widget_Slider(col1,  $
      UNAME='ANV_SLIDER_select' ,YOFFSET=195  $
      , maximum = anv_n_files, scroll = 1 $
      ,SCR_XSIZE=182)

  ANV_BUT_KEEP = Widget_Button(col1,  SCR_YSIZE = 24, SCR_XSIZE=70, $
      UNAME='ANV_BUT_KEEP' ,XOFFSET=200 ,YOFFSET=180  $
      ,/ALIGN_CENTER ,VALUE='Keep')

  ANV_BUT_DISCARD = Widget_Button(col1,  SCR_YSIZE = 24, SCR_XSIZE=70, $
      UNAME='ANV_BUT_DISCARD' ,XOFFSET=200 ,YOFFSET=210  $
      ,/ALIGN_CENTER ,VALUE='Discard')

  ANV_BUT_DISMISS = Widget_Button(col1, UNAME='ANV_BUT_DISMISS'  $
      ,XOFFSET=200 ,YOFFSET=290, SCR_YSIZE = 36, SCR_XSIZE=70   $
      ,/ALIGN_CENTER ,VALUE='Dismiss')

; Load data into widget
WIDGET_CONTROL,ANV_TEXT_folder, Set_VALUE = folder
t = ax_name(anv_filelist(anv_current_file))
WIDGET_CONTROL,ANV_TEXT_filename, Set_VALUE = t(1)

; set preview_flag to 1 to force 2d or 2d data to be dispayed
preview_flag = 1

ptycho_flag=0

; ------ OK all set, lets generate the widget

  Widget_Control, /REALIZE, BASE_ANV

  XManager, 'AX_NEXUS_VIEWER', BASE_ANV, /NO_BLOCK

 AX_NEXUS_VIEWER_ID = BASE_ANV

; ----------- get info needed to access the draw_window (only after REALIZE !)
widget_control, ANV_DRAW, GET_value=ANV_NexusImg

; ----- load first file
 ANV_UPDATE   ;, BASE_ANV

end




pro ax_nexus_viewer, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_
@axis_com
@nexus_com
on_error,2

  BASE_AX_NEXUS_VIEWER, GROUP_LEADER=wGroup, _EXTRA=_VWBExtra_

; set preview_flag to 1 to force 2d or 2d data to be dispayed
preview_flag = 1

; empty stub to load
end
