; Copyright (c) 1998-2014 A.P. Hitchcock  All rights reserved
;+
;NAME:
;  STACK_LINE_Io
;
;LAST CHANGED: ----------------------------------- 12 May 2014
;
; PURPOSE:
;	This function normalizes a stack on a row by row basis
; When used for slow stacks, with clear Io region at the same <X-range> in all rows,
; this provides a more accurate correction of Io changes (e.g' from ring current decay)
;
; CATEGORY:
;	stack processing; stand alone operation or from axis2000
;
; CALLING SEQUENCE:
;	RESULT = STACK_LINE_Io()
;
; INPUTS: none
;
; KEYWORDS: none
;
; OUTPUTS: 	routine stores the OD-converted stack and
;           returns the %-deviation of Io from <Io> on an energy-by-energy basis
;
; COMMON BLOCKS:
; AXIS_COM	standard set of common blocks
; stack_process_com
; BSIF_com
; volume_data, image_stack
;
; RESTRICTIONS: none
;
; MODIFICATION HISTORY:
; (12-May-14 aph) first version - requested by Slava for AFCC ionomer mapping at CLS
;-

FUNCTION stack_line_io
@axis_com
@stack_process_com
COMMON volume_data, image_stack
@bsif_com

on_error,2

; ---- read in Io stack (a truncated version of the full stack  ----
stack_io   = pickfile2(/READ, title='select Io stack', FILTER='*.ncb',/LPATH, DEFPATH=defpath)
stack_rb, stack_io
io=image_stack
ts = size(io)
io_ncols=ts(1)
io_nrows=ts(2)
io_npts=ts(3)
; print, 'Io: ncols, nrows, npts ', io_ncols, io_nrows, io_npts

; ---- read in transmission stack ----
stack_data = pickfile2(/READ, title='select transmission stack', FILTER='*.ncb',/LPATH, DEFPATH=defpath)
stack_rb, stack_data
stack=image_stack
ts = size(stack)
stack_ncols=ts(1)
stack_nrows=ts(2)
stack_npts=ts(3)
; print, 'STACK: ncols, nrows, npts ', stack_ncols, stack_nrows, stack_npts

; ------ check for data file compatibility ---
if io_nrows NE stack_nrows or io_npts NE stack_npts then begin
	axis_log, '# of rows or # of energies of Io stack  differ from those of the transmission stack'
endif

; ----- generate Io image-spectrum by averaging each row of the Io stack ---
Io_image = fltarr(stack_npts, stack_nrows)
for i = 0, stack_npts - 1 do begin
	for j = 0, io_nrows - 1 do begin
		t =  moment(io(*,j,i))
;		print, j, '  ', t(0)
		Io_image(i,j) = t(0)
	endfor
endfor

; generate axis format image  of the Io signal, in squae format
y = findgen(stack_nrows)
nx = io_nrows
x = findgen(nx)
block = floor(nx/io_npts)
d = fltarr(nx, stack_nrows)
for i = 0, io_npts-1 do begin
	line1 = i*block
	for j = 0, block-1 do d(line1+j, *) = Io_image(i,*)
endfor
for j = line1+block, nx-1 do d(j,*)=Io_image(io_npts-1,*)

; ----- convert to %-change over the line
a=total(d,2,/double)/stack_nrows
for i=0,stack_nrows-1 do d(*,i)=100.*d(*,i)/a
; generate aXis format image to return to aXis2000
t=ax_name(stack_data)
text = t(1) + '    %-devation from <Io>'
tmp = {t:'2d', x:x, y:y, xl:'line-Io', e: ev(0), yl:'rows', d: d, dl:text}

; ------- convert to OD using line-by-line Io
image_stack = fltarr(stack_ncols, stack_nrows, stack_npts)
for k = 0, stack_npts -1 do begin
	for i = 0, stack_ncols - 1 do begin
		for j = 0, stack_nrows - 1 do image_stack(i,j,k)=alog(Io_image(k,j)/stack(i,j,k))
	endfor
endfor

; ----- write out converted stack -----------------
file = pickfile2(/READ, TITLE = 'OD stack filename', FILTER='*.ncb', /LPATH, DEFPATH=defpath)
if strlen(file) NE 0 then begin
	t = ax_name(file)
	filename = string(t(0)) + string(t(1)) + '.ncb'
	stack_wb, filename
endif

ax_wait,/off		; turn off hourglass (sometimes sticks on)
return, tmp

end

